/*
 * HAMULIGHT L2046 transformer PI PICO with CC1101 TI transceiver
 * PI PICO is connected with SPI bus to CC1101 transceiver module
 * PI PICO uses a bitstream output and bitstream input connection
 * */
#include <stdio.h>
#include <string.h>
#include "pico/stdlib.h"
#include "hardware/pwm.h"
#include "hardware/spi.h"
#include "hardware/flash.h"
#include "hardware/sync.h"

//GPIO pins defined
#define PWM_OUT 26      // gpio26 output pin of PWM, just to check if PWM is running
#define TEST_OUT 0      // GPIO0 output pin in the program can be used to measure some timing
#define RF_DATA_IN 28   // GPIO28 as input sample pin cumming from RX-CC1101
#define RF_pin 22       // GPIO22 Output pin to send bitstream to TX-CC1101
#define LED_PIN 25      // GPIO25 output pin, which is connected to to PI PICO onboard LED

#define BIT_BUFFER_LENGTH 4736


//sample buffer of 100x32 bit words (32x100 samples of 50us each sample = 160ms)
//
// a total of 20 words is activly used is the program 20x32= 640 samples x 50us = 32ms
//
// a remote frame is 1600us preamble + 1200us high + 1200us low + 32x800us = 25.6ms
//
// a sample buffer of 32ms should be enough and is proven to be enough to capture a frame from
// HUMALIGHT remote control
//
#define WORD_BUFFER_LENGTH 100


// time in us for a ONE bit high/low timing and ZERO bit high/low timing
#define ONE_HIGH 200
#define ONE_LOW 597
#define ZERO_HIGH 600
#define ZERO_LOW 197

// flash address offset of last sector (4kB) in flash memory to store remote serial and data
// each 4kB sector is devided into pages of 256 bytes, so each sector is 16 pages
#define FLASH_TARGET_OFFSET (PICO_FLASH_SIZE_BYTES - FLASH_SECTOR_SIZE)

//SPI Pins defined
#define CS_PIN 5
#define SCK_PIN 2
#define MOSI_PIN 3
#define MISO_PIN 4

bool DATA_IN_BITS[BIT_BUFFER_LENGTH];
uint DATA_IN_COUNTER = 0;
uint32_t RECEIVE_32_BIT[BIT_BUFFER_LENGTH/32];
uint32_t RECEIVE_TEMP=0;
uint32_t REMOTE_32_BIT_WORD=0;
char REMOTE_SERIAL_HEX[5]="A7E4\0";
char REMOTE_SERIAL_FLASH[5] = "0001\0";
char REMOTE_DATA_FLASH[5] = "0000\0";
char REMOTE_DATA_HEX[5] = "0000\0";
uint16_t REMOTE_SERIAL_16 = 0x0000;
uint16_t REMOTE_DATA_16 = 0x0000;
uint16_t REMOTE_DIM_16 = 0x0000;
uint16_t DIM_HEX_16_BIT = 0xB000;
char DIM_HEX[5] = "0000\0";
uint32_t DATA_IN_BUFFER[WORD_BUFFER_LENGTH];
int BIT_POINTER = 31;
uint16_t WORD_POINTER=0;

char REMOTE_SERIAL_BITS_CHAR[17]= "1010011111100100\0";// A7E4
char DIM_PERCENT[4] = "100\0";
uint32_t TRANSMIT_32_BIT = 0;

uint8_t msg[10];
uint8_t SPI_RX_DATA[0x64];
uint32_t *POINTER_TO_FLASH, ADDRESS_FLASH_STORE;
uint32_t FLASH_VALUE;
// One flash page (256byte) buffer, as one page is the smallest number of bytes which can be written to flash
uint8_t FLASH_BUFFER[FLASH_PAGE_SIZE/sizeof(uint8_t)];
 
// Write 1 byte to the specified register CC1101
void reg_write( spi_inst_t *spi, 
                const uint cs, 
                const uint8_t reg, 
                const uint8_t data) 
{
            
    // Construct message (set ~W bit low, MB bit low)
    msg[0] = 0x00 | reg;
    msg[1] = data;

    // Write to register
    gpio_put(cs, 0);
    sleep_us(10);
    spi_write_blocking(spi, msg, 2);
    gpio_put(cs, 1);
}

int reg_write_burst_patatable( spi_inst_t *spi, 
                const uint cs, 
                const uint8_t reg, 
                uint8_t *buf,
                const uint8_t nbytes) 
{
  
    int num_bytes_read = 0;
    uint8_t mb = 0;  
    // Determine if multiple byte (MB) bit should be set
    if (nbytes < 1) {
        return -1;
    } else if (nbytes == 1) {
        mb = 0;
        msg[0]=reg&0xCF;
    } else {
        mb = 1;
        msg[0]=reg|0x40;
    }          
    // Construct message (set ~W bit low, MB bit low)
    msg[0] = msg[0] &0x7f;
    msg[1] = buf[0];
    msg[2] = buf[1];
    
    // Write to register
    gpio_put(cs, 0);
    sleep_us(10);
    spi_write_blocking(spi, msg, 9);
    gpio_put(cs, 1);
}

// Read byte(s) from specified register. If nbytes > 1, read from consecutive
// registers.
int reg_read(  spi_inst_t *spi,
                const uint cs,
                const uint8_t reg,
                uint8_t *buf,
                const uint8_t nbytes) 
{

    int num_bytes_read = 0;
    uint8_t mb = 0;

    // Determine if multiple byte (MB) bit should be set
    if (nbytes < 1) {
        return -1;
    } else if (nbytes == 1) {
        mb = 0;
        
    } else {
        mb = 1;
        
    }

    // Construct message (set ~W bit high)
    msg[0] = 0x80 | (mb << 6) | reg;

    // Read from register
    gpio_put(cs, 0);
    spi_write_blocking(spi, msg, 1);
    num_bytes_read = spi_read_blocking(spi, 0, buf, nbytes);
    gpio_put(cs, 1);

    return num_bytes_read;
}
//first preamble of bitstream to be send toHuma light
void first_pre_amble(void) 
{

    gpio_put(RF_pin, 1);
    sleep_us(200);
    gpio_put(RF_pin, 0);
    sleep_us(200);
    gpio_put(RF_pin, 1);
    sleep_us(200);
    gpio_put(RF_pin, 0);
    sleep_us(200);
    gpio_put(RF_pin, 1);
    sleep_us(200);
    gpio_put(RF_pin, 0);
    sleep_us(200);
    gpio_put(RF_pin, 1);
    sleep_us(1200);
    gpio_put(RF_pin, 0);
    sleep_us(1200);

}
// 2nd and the rest of preambles used
void normal_pre_amble(void) 
{

    gpio_put(RF_pin, 1);
    sleep_us(200);
    gpio_put(RF_pin, 0);
    sleep_us(200);
    gpio_put(RF_pin, 1);
    sleep_us(200);
    gpio_put(RF_pin, 0);
    sleep_us(200);
    gpio_put(RF_pin, 1);
    sleep_us(200);
    gpio_put(RF_pin, 0);
    sleep_us(200);
    gpio_put(RF_pin, 1);
    sleep_us(200);
    gpio_put(RF_pin, 0);
    sleep_us(200);
    gpio_put(RF_pin, 1);
    sleep_us(1200);
    gpio_put(RF_pin, 0);
    sleep_us(1200);
}

void transmit(uint32_t WORD_32_BIT_TO_SEND) //char v_sd[33]) {
{
    int bit_count = 0;
    int packet_count = 1;
    uint32_t TEMP_32_BIT;

   
    
    packet_count=1;
    for(packet_count = 1; packet_count <= 20 ; packet_count++)
    {
        TEMP_32_BIT=WORD_32_BIT_TO_SEND;
        normal_pre_amble();
        for(bit_count = 0; bit_count<32; bit_count++)
        {
            for(bit_count = 0; bit_count<32; bit_count++)
            {
                if((TEMP_32_BIT&0x80000000)== 0)
                {
                    gpio_put(RF_pin, 1);
                    sleep_us(ZERO_HIGH);
                    gpio_put(RF_pin, 0);
                    sleep_us(ZERO_LOW);
                    TEMP_32_BIT=TEMP_32_BIT<<1;
                }
                else
                {
                    gpio_put(RF_pin, 1);
                    sleep_us(ONE_HIGH);
                    gpio_put(RF_pin, 0);
                    sleep_us(ONE_LOW); 
                    TEMP_32_BIT=TEMP_32_BIT<<1;
                }
            }
            
        }
    }

}

void transmit_dim(uint32_t WORD_32_BIT_TO_SEND) //char v_sd[33]) {
{
    int bit_count = 0;
    int packet_count = 1;
    uint32_t TEMP_32_BIT;

    packet_count=1;
    for(int i = 1; i<=2; i++)
    {
        sleep_ms(45);
        for(packet_count = 1; packet_count <= 1 ; packet_count++)
        {
             first_pre_amble();
            TEMP_32_BIT=WORD_32_BIT_TO_SEND;
            for(bit_count = 0; bit_count<32; bit_count++)
            {
                if((TEMP_32_BIT&0x80000000)== 0)
                {
                    gpio_put(RF_pin, 1);
                    sleep_us(ZERO_HIGH);
                    gpio_put(RF_pin, 0);
                    sleep_us(ZERO_LOW);
                    TEMP_32_BIT=TEMP_32_BIT<<1;
                }
                else
                {
                    gpio_put(RF_pin, 1);
                    sleep_us(ONE_HIGH);
                    gpio_put(RF_pin, 0);
                    sleep_us(ONE_LOW); 
                    TEMP_32_BIT=TEMP_32_BIT<<1;
                }
            }
            TEMP_32_BIT=WORD_32_BIT_TO_SEND;
            normal_pre_amble();
            for(bit_count = 0; bit_count<32; bit_count++)
            {
                if((TEMP_32_BIT&0x80000000)== 0)
                {
                    gpio_put(RF_pin, 1);
                    sleep_us(ZERO_HIGH);
                    gpio_put(RF_pin, 0);
                    sleep_us(ZERO_LOW);
                    TEMP_32_BIT=TEMP_32_BIT<<1;
                }
                else
                {
                    gpio_put(RF_pin, 1);
                    sleep_us(ONE_HIGH);
                    gpio_put(RF_pin, 0);
                    sleep_us(ONE_LOW); 
                    TEMP_32_BIT=TEMP_32_BIT<<1;
                }
            }
        }
    }
    
}

void transmit_fast_dim(uint32_t WORD_32_BIT_TO_SEND) //char v_sd[33]) {
{
    int bit_count = 0;
    int packet_count = 1;
    uint32_t TEMP_32_BIT;

    packet_count=1;
    for(int i = 1; i<=2; i++)
    {
        sleep_ms(45);
        for(packet_count = 1; packet_count <=1 ; packet_count++)
        {
            TEMP_32_BIT=WORD_32_BIT_TO_SEND;
            first_pre_amble();
            for(bit_count = 0; bit_count<32; bit_count++)
            {
          
                if((TEMP_32_BIT&0x80000000)== 0)
                {
                    gpio_put(RF_pin, 1);
                    sleep_us(ZERO_HIGH);
                    gpio_put(RF_pin, 0);
                    sleep_us(ZERO_LOW);
                    TEMP_32_BIT=TEMP_32_BIT<<1;
                }
                else
                {
                    gpio_put(RF_pin, 1);
                    sleep_us(ONE_HIGH);
                    gpio_put(RF_pin, 0);
                    sleep_us(ONE_LOW); 
                    TEMP_32_BIT=TEMP_32_BIT<<1;
                }
            }
            TEMP_32_BIT=WORD_32_BIT_TO_SEND;
            normal_pre_amble();
            for(bit_count = 0; bit_count<32; bit_count++)
            {
               
                if((TEMP_32_BIT&0x80000000)== 0)
                {
                    gpio_put(RF_pin, 1);
                    sleep_us(ZERO_HIGH);
                    gpio_put(RF_pin, 0);
                    sleep_us(ZERO_LOW);
                    TEMP_32_BIT=TEMP_32_BIT<<1;
                }
                else
                {
                    gpio_put(RF_pin, 1);
                    sleep_us(ONE_HIGH);
                    gpio_put(RF_pin, 0);
                    sleep_us(ONE_LOW); 
                    TEMP_32_BIT=TEMP_32_BIT<<1;
                }
            }
        }
    }
   
}


void bool_to_32_bit_words()
{
    for(int i=0; i<(BIT_BUFFER_LENGTH/32); i++)
    {
        for(int j=0; j<32; j++)
        {
            RECEIVE_32_BIT[i]=RECEIVE_32_BIT[i]<<1;
            if(DATA_IN_BITS[i*32+j]==true)
            {
                RECEIVE_32_BIT[i]=RECEIVE_32_BIT[i]|0x00000001;
            }
            else
            {
                RECEIVE_32_BIT[i]=RECEIVE_32_BIT[i]&0xFFFFFFFE;
            }
   
        }
    }
}

bool find_preamble()
{
    // preamble is 4 samples (4x50us=200us) high and 4 samples low. 4 times high/low of 200us/200us is preamble signal 
    // HUMALIGHT remote. 1200us high(24 samples) and 1200us low (24 samples) follows the the 4 times high/low signal (4x8 samples=32 samples)

    // total number of samples from preamble including 1200us/high/ 1200us low is 80 samples in totalbefore 16 bit serial and 16 bit data is available in 
    // input sample bit buffer  
    //
    // each bit is remote signal is 800us. SO each bit is 16 samples. 
    //
    //
    // i stated a ONE bit is 200us highand 600us low. A ZERO bit is 600us high and 200us low
    //
    if(RECEIVE_32_BIT[0]==0xF0F0F0F0)
    {
        if(RECEIVE_32_BIT[1]==0xFFFFFF00)
               {
                    RECEIVE_TEMP=RECEIVE_32_BIT[2]&0xffff0000;
                    if(RECEIVE_TEMP ==0x00000000)
                    {
                        return true;
                    }
                    else
                    {
                        return false;
                    }
               }
    }
    else
    {
        for(int y = 0; y<BIT_BUFFER_LENGTH; y++)
        {
            for(int i = 0; i<(BIT_BUFFER_LENGTH/32)-1; i++)
            {
                RECEIVE_32_BIT[i]=RECEIVE_32_BIT[i]<<1;
                RECEIVE_TEMP=RECEIVE_32_BIT[i+1]&0x80000000;
                if(RECEIVE_TEMP)
                {
                    RECEIVE_32_BIT[i]=RECEIVE_32_BIT[i]|0x00000001;
                }
                else
                {
                    RECEIVE_32_BIT[i]=RECEIVE_32_BIT[i]&0xFFFFFFFE;
                }
                        }
            if(RECEIVE_32_BIT[0]==0xF0F0F0F0)
            {
               if(RECEIVE_32_BIT[1]==0xFFFFFF00)
               {
                    RECEIVE_TEMP=RECEIVE_32_BIT[2]&0xffff0000;
                    if(RECEIVE_TEMP ==0x00000000)
                    {
                        return true;
                    }
                    else
                    {
                        return false;
                    }
               }
               else
               {
                return false;
               }
            
            }
        }
        return false;
    }
}

uint32_t number_of_one_bits(uint32_t CHECK_WORD)
{
    uint32_t ONE_BIT_COUNTER=0;
    uint32_t TEMP_CHECK_WORD = 0;
    ONE_BIT_COUNTER=0;

    for(int i=0; i<16;i++)
    {
        TEMP_CHECK_WORD=CHECK_WORD&0x00000001;

        if(TEMP_CHECK_WORD>0)
        {
            ONE_BIT_COUNTER++;

        }
        CHECK_WORD=CHECK_WORD>>1;
    }
    return ONE_BIT_COUNTER;
}

uint32_t receive_bits_to_one_word(void)
{
    uint32_t TEMP_WORD1=0;
    uint32_t TEMP_WORD2=0;
    uint32_t REMOTE_WORD=0;
    uint32_t ONES_TEMP_WORD1=0;
    uint32_t ONES_TEMP_WORD2=0;

    TEMP_WORD1=TEMP_WORD2=REMOTE_WORD=ONES_TEMP_WORD1=ONES_TEMP_WORD2=0;

    for(int i=0; i<16; i++)
    {
        ONES_TEMP_WORD1=ONES_TEMP_WORD2=0;
        TEMP_WORD1=DATA_IN_BUFFER[i];
        TEMP_WORD2=DATA_IN_BUFFER[i]&0xFFFF0000;
        TEMP_WORD2=TEMP_WORD2>>16;
        TEMP_WORD1=TEMP_WORD1&0x0000FFFF;

        ONES_TEMP_WORD1=number_of_one_bits(TEMP_WORD1);
        ONES_TEMP_WORD2=number_of_one_bits(TEMP_WORD2);
        //printf("test\n");
        if(ONES_TEMP_WORD2 > 11)
        {
            REMOTE_WORD=REMOTE_WORD&0xFFFFFFFE;
        }
        else
        {
            REMOTE_WORD=REMOTE_WORD|0x00000001;
        }
        REMOTE_WORD=REMOTE_WORD<<1;

        if(ONES_TEMP_WORD1 > 11)
        {
            REMOTE_WORD=REMOTE_WORD&0xFFFFFFFE;
        }
        else
        {
            REMOTE_WORD=REMOTE_WORD|0x00000001;
        }
        if(i<15)
        {
            REMOTE_WORD=REMOTE_WORD<<1;
        }
    }
   
    return REMOTE_WORD;
}

void hex_to_string(uint32_t REMOTE_32_BIT)
{
    uint32_t REMOTE[8]={0,0,0,0,0,0,0,0};
    uint16_t j = 0;

    for(int i=0;i<8; i++)
    {
        REMOTE[i] = (REMOTE_32_BIT>>(i*4))&0x0000000f;
    }

    j=0;
    for(int i=7; i>=0; i--)
    {
        if(i>=4)
        {
            if(REMOTE[i]<=9)
            {
                REMOTE_SERIAL_HEX[j]=(char)REMOTE[i]+48;
            }
            else
            {
                REMOTE_SERIAL_HEX[j]=(char)REMOTE[i]+55;
            }
            j=j+1;
            if(j>=4)
            {
                j=0;
            }
        }
        else
        {
           if(REMOTE[i]<=9)
            {
                REMOTE_DATA_HEX[j]=(char)REMOTE[i]+48;
            }
            else
            {
                REMOTE_DATA_HEX[j]=(char)REMOTE[i]+55;
            }
            j=j+1; 
        }
    }
    
}

void shift_input_buffer_1_bit()
{
    uint32_t TEMP_AND = 0;

    for (int i=0; i<20; i++)
    {
        DATA_IN_BUFFER[i]=DATA_IN_BUFFER[i]<<1;
        if(i<19)
        {
            TEMP_AND=DATA_IN_BUFFER[i+1] & 0x80000000;
        }
        if(TEMP_AND>0)
        {
            DATA_IN_BUFFER[i]=DATA_IN_BUFFER[i]|0x00000001;
        }
        else
        {
            DATA_IN_BUFFER[i]=DATA_IN_BUFFER[i]&0xFFFFFFFE;
        }
    }
    
}

void on_pwm_wrap()
{
    bool INPUT_BIT_VALUE = false;
    
    gpio_put(TEST_OUT,true);
    //sample RF data input pin from receive pin CC1101 tranceiver
    INPUT_BIT_VALUE=gpio_get(RF_DATA_IN);
    //printf("input pin value %d\n", INPUT_BIT_VALUE);
    if((BIT_POINTER<0)&&(WORD_POINTER<19))
    { 
        BIT_POINTER=31;
        WORD_POINTER=WORD_POINTER+1;
    }
    // every time the PWM interrupt routine is entered (cycle time 50us) check if HUMA light pre-amblesignal
    // is found at begin of input bit buffer
    //
    if(DATA_IN_BUFFER[0]!=0xF0F0F0F0)
    {
       
        if(INPUT_BIT_VALUE)
        {
            DATA_IN_BUFFER[WORD_POINTER]=DATA_IN_BUFFER[WORD_POINTER] | 1<<BIT_POINTER;
            //printf("bit: %d WORD: %d value: %d\n",BIT_POINTER,WORD_POINTER, INPUT_BIT_VALUE);
        }
        else
        {
            DATA_IN_BUFFER[WORD_POINTER]=DATA_IN_BUFFER[WORD_POINTER] & ~( 1<<BIT_POINTER);
            //printf("bit: %d WORD: %d value: %d\n",BIT_POINTER,WORD_POINTER, INPUT_BIT_VALUE);
        }
        
    }
    else
    {
        //
        // THis routine will be done when the HUMALIGHT pre-amble signal is received
        
        //printf("found pre-amble \n");
        // gpio_put(TEST_OUT,true);
        
        // shift input buffer with 80 samples(50us per sample) to get the HUMALIGHT serial number and data bites are in the begin
        // of the input bit buffer
        //
        for(int i=0; i<80; i++)
        {
            shift_input_buffer_1_bit();
        }
            
        REMOTE_32_BIT_WORD=receive_bits_to_one_word();
        hex_to_string(REMOTE_32_BIT_WORD);
        printf("i did find preamble and remote SERIAL %s remote DATA %s \n",REMOTE_SERIAL_HEX, REMOTE_DATA_HEX);
        for(int i = 0; i<20; i++)
        {
            DATA_IN_BUFFER[i]=0;
        }
    }
    if((BIT_POINTER>=0)&&(WORD_POINTER<19))
    {
        BIT_POINTER=BIT_POINTER-1;
    }
    if(WORD_POINTER==19)
    {
        BIT_POINTER=BIT_POINTER-1;
        if(BIT_POINTER<0)
        {
            BIT_POINTER=0;
            shift_input_buffer_1_bit();
        }
        //printf("recieve buffer filled\n");
    }
    pwm_clear_irq(pwm_gpio_to_slice_num(PWM_OUT));
    
    gpio_put(TEST_OUT,false);
}

uint16_t string_to_integer(void)
{
    uint16_t DIM_INTEGER_100 = 0;
    uint16_t DIM_INTEGER_10 = 0;
    uint16_t DIM_INTEGER_1 = 0 ;
    uint16_t NUMBER_OF_CHAR_IN_DIM_LEVEL_STRING =0;
    
    for(NUMBER_OF_CHAR_IN_DIM_LEVEL_STRING=1; (NUMBER_OF_CHAR_IN_DIM_LEVEL_STRING<4)&&(DIM_PERCENT[NUMBER_OF_CHAR_IN_DIM_LEVEL_STRING]!= 0x00); NUMBER_OF_CHAR_IN_DIM_LEVEL_STRING++);

    if(DIM_PERCENT[0]== 0x00)
    {
        return 0;
    }
    if(NUMBER_OF_CHAR_IN_DIM_LEVEL_STRING==1)
    {
        if(DIM_PERCENT[0]=='0')
        {
            return 0;
        }
        DIM_INTEGER_100 = 0;
        DIM_INTEGER_10 =0;
        DIM_INTEGER_1 = DIM_PERCENT[0]-0x30;
        return (DIM_INTEGER_100*100+DIM_INTEGER_10*10+DIM_INTEGER_1); 
    }
    if(NUMBER_OF_CHAR_IN_DIM_LEVEL_STRING==2)
    {
        DIM_INTEGER_100 = 0;
        DIM_INTEGER_10 = DIM_PERCENT[0]-0x30;
        DIM_INTEGER_1 = DIM_PERCENT[1]-0x30;
        return (DIM_INTEGER_100*100+DIM_INTEGER_10*10+DIM_INTEGER_1);
    }
    if(NUMBER_OF_CHAR_IN_DIM_LEVEL_STRING==3)
    {
        return 100;
    }
}

uint16_t value_to_16_bit_dim_level(uint16_t DIM_LEVEL)
{
    if(DIM_LEVEL<=75)
    {
        if(DIM_LEVEL==75)
        {
            return 0xFBFF;
        }
        else
        {
            DIM_LEVEL=DIM_LEVEL*((0xFBFF-0xB000)/75);
            DIM_LEVEL=DIM_LEVEL+0xB000;
            return DIM_LEVEL;   
        }
    }
    else
    {
        if(DIM_LEVEL==100)
        {
            DIM_LEVEL=0x9FFF;
            return DIM_LEVEL;
        }
        else
        {
            DIM_LEVEL=DIM_LEVEL-75;
            DIM_LEVEL=DIM_LEVEL*((0xA000-0x8000)/25);
            DIM_LEVEL=DIM_LEVEL+0x8000;
            return DIM_LEVEL;
        }
    }
}

uint32_t serial_and_dim_16_bit_to_32_bit(uint16_t DIM_16_BIT, uint16_t REMOTE_SERIAL)
{
    
    uint32_t TRANSMIT_32_TEMP;

    TRANSMIT_32_TEMP = 0x00000000;
    TRANSMIT_32_TEMP=TRANSMIT_32_TEMP+REMOTE_SERIAL;
    TRANSMIT_32_TEMP=TRANSMIT_32_TEMP<<16;
    TRANSMIT_32_TEMP=TRANSMIT_32_TEMP+DIM_16_BIT;
    return TRANSMIT_32_TEMP;
}

uint16_t hex_string_to_16_bit(char *HEX_STRING_4_DIGIT)
{
    uint16_t HEX_TEMP = 0x0000;
    uint16_t HEX_TEMP_16_BIT = 0x0000;

   
    HEX_TEMP_16_BIT = 0x0000;
    for(int i=0; i<4; i++)
    {
        if((HEX_STRING_4_DIGIT[i] >= 'A')&&(HEX_STRING_4_DIGIT[i] <= 'F'))
        {
            HEX_TEMP = HEX_STRING_4_DIGIT[i]-55;
            HEX_TEMP_16_BIT=HEX_TEMP_16_BIT + HEX_TEMP;
            if(i<3)
            {
                HEX_TEMP_16_BIT = HEX_TEMP_16_BIT<<4;
            }
        }
        if((HEX_STRING_4_DIGIT[i] >= '0')&&(HEX_STRING_4_DIGIT[i] <= '9'))
        {
            HEX_TEMP = HEX_STRING_4_DIGIT[i]-48;
            HEX_TEMP_16_BIT=HEX_TEMP_16_BIT + HEX_TEMP;
            if(i<3)
            {
                HEX_TEMP_16_BIT = HEX_TEMP_16_BIT<<4;
            }
        }
    }
    return HEX_TEMP_16_BIT;
}

void wait_for_settle_transmit_mode(void)
{
    do
    {
        reg_read(spi0, CS_PIN, 0x3D, SPI_RX_DATA, 1);
        SPI_RX_DATA[0]=SPI_RX_DATA[0]&0xF0;
    }  while(SPI_RX_DATA[0]!=0x20);

}

void wait_for_settle_receive_mode(void)
{
    do
    {
        reg_read(spi0, CS_PIN, 0x3D, SPI_RX_DATA, 1);
        SPI_RX_DATA[0]=SPI_RX_DATA[0]&0xF0;
    }  while(SPI_RX_DATA[0]!=0x10);
}

uint8_t char_to_BCD(char CHARACTER_TO_TRANSFORM)
{
    if((CHARACTER_TO_TRANSFORM>='A') &&(CHARACTER_TO_TRANSFORM<='F'))
    {
        CHARACTER_TO_TRANSFORM=CHARACTER_TO_TRANSFORM-55;
    }
     if((CHARACTER_TO_TRANSFORM>='0') &&(CHARACTER_TO_TRANSFORM<='9'))
    {
        CHARACTER_TO_TRANSFORM=CHARACTER_TO_TRANSFORM-48;
    }
    return CHARACTER_TO_TRANSFORM;
}
void read_stored_remote_info()
{
    ADDRESS_FLASH_STORE=XIP_BASE + FLASH_TARGET_OFFSET;
    POINTER_TO_FLASH=(uint32_t *)ADDRESS_FLASH_STORE;
    FLASH_VALUE=*(POINTER_TO_FLASH);
    REMOTE_SERIAL_FLASH[0]=FLASH_VALUE&0x000000ff;
    REMOTE_SERIAL_FLASH[1]=(FLASH_VALUE&0x0000FF00)>>8;
    REMOTE_SERIAL_FLASH[2]=(FLASH_VALUE&0x00FF0000)>>16;
    REMOTE_SERIAL_FLASH[3]=(FLASH_VALUE&0xFF000000)>>24;
    REMOTE_SERIAL_FLASH[4]=0x00;
    POINTER_TO_FLASH=POINTER_TO_FLASH+1;
    FLASH_VALUE=*(POINTER_TO_FLASH);
    REMOTE_DATA_FLASH[0]=FLASH_VALUE&0x000000ff;
    REMOTE_DATA_FLASH[1]=(FLASH_VALUE&0x0000FF00)>>8;
    REMOTE_DATA_FLASH[2]=(FLASH_VALUE&0x00FF0000)>>16;
    REMOTE_DATA_FLASH[3]=(FLASH_VALUE&0xFF000000)>>24;
    REMOTE_DATA_FLASH[4]=0x00;
    
    REMOTE_SERIAL_16=0x0000;
    for(int i=0; i<4; i++)
    {
        REMOTE_SERIAL_16=REMOTE_SERIAL_16|(char_to_BCD(REMOTE_SERIAL_FLASH[i]));
        if(i<3)
        {
            REMOTE_SERIAL_16=REMOTE_SERIAL_16<<4;
        }
    }
    REMOTE_DATA_16=0x0000;
    for(int i=0; i<4; i++)
    {
        REMOTE_DATA_16=REMOTE_DATA_16|(char_to_BCD(REMOTE_DATA_FLASH[i]));
        if(i<3)
        {
            REMOTE_DATA_16=REMOTE_DATA_16<<4;
        }
    }
}

int main() 
{

    char recieve_char[25] = "\0\0";
    char send_char[25] = "\0";
    char on_command[4] = "VON\0";
    char off_command[5] = "VOFF\0";
    char dim_100[5] = "B100\0";
    char dim_75[4] = "B75\0";
    char dim_50[4] = "B50\0";
    char dim_25[4] = "B25\0";
    char dim_xx[4] = "DIM\0";
    char hex_xx[4] = "HEX\0";
    char down[5] = "DOWN\0";
    char test[2] = "\0\0";
    char up[3] = "UP\0";
    char learn[6] = "LEARN\0";
    char setting[9] = "SETTINGS\0";
    int string_equal = 1;
    int teller = 0;
    uint slice = 0;
    uint channel = 0;
    uint16_t DIM_LEVEL = 0;
    uint16_t DIM_16_BIT = 0;

    char VALUE0, VALUE1, VALUE2, VALUE3, VALUE4, VALUE5, VALUE6, VALUE7;
    
    // CC1101 register setting from TI RF tool. first byte is register, second byte is data of the register
    uint8_t CC1101_REG_SETTINGS[] = {0x00, 0x0D, //SMARTRF_SETTING_IOCFG2
                                    0x02, 0x0C, //SMARTRF_SETTING_IOCFG0
                                    0x03, 0x47, //SMARTRF_SETTING_FIFOTHR
                                    0x08, 0x32, //SMARTRF_SETTING_PKTCTRL0 0x32
                                    0x0B, 0x06, //SMARTRF_SETTING_FSCTRL1
                                    0x0D, 0x10, //SMARTRF_SETTING_FREQ2 0x10
                                    0x0E, 0xB0, //SMARTRF_SETTING_FREQ1 0xB0
                                    0X0F, 0x71, //SMARTRF_SETTING_FREQ0 0x71
                                    0x10, 0xC9, //SMARTRF_SETTING_MDMCFG4 0xF7
                                    0x11, 0x93, //SMARTRF_SETTING_MDMCFG3
                                    0x12, 0x30, //SMARTRF_SETTING_MDMCFG2 
                                    0x15, 0x34, //SMARTRF_SETTING_DEVIATN 0x15
                                    0x17, 0x0B, //SMARTRF_SETTING_MCSM1 0x0B thershold signal
                                    0x18, 0x18, //SMARTRF_SETTING_MCSM0
                                    0x19, 0x16, //SMARTRF_SETTING_FOCCFG
                                    0x20, 0xFB, //SMARTRF_SETTING_WORCTRL
                                    0x22, 0x01, //SMARTRF_SETTING_FREND0 0x11 01
                                    0x23, 0xE9, //SMARTRF_SETTING_FSCAL3
                                    0x24, 0x2A, //SMARTRF_SETTING_FSCAL2
                                    0x25, 0x00, //SMARTRF_SETTING_FSCAL1
                                    0x26, 0x1F, //SMARTRF_SETTING_FSCAL0
                                    0x2C, 0x81, //SMARTRF_SETTING_TEST2 
                                    0x2D, 0x35, //SMARTRF_SETTING_TEST1
                                    0X2E, 0x09};//SMARTRF_SETTING_TEST0


    
    
    //
    //SET CPU clock to 125 MHz
    set_sys_clock_khz(125000, 1);
    stdio_init_all();

    // Initialise UART 0
    uart_init(uart0, 115200);
    
    //init GPIO for PI PICO onboard LED PIN
    gpio_init(LED_PIN);
    gpio_set_dir(LED_PIN, GPIO_OUT);
    gpio_set_drive_strength(LED_PIN, GPIO_DRIVE_STRENGTH_4MA);
    gpio_set_slew_rate(LED_PIN, GPIO_SLEW_RATE_SLOW);
    
    //PI PICO onboard LED ON
    gpio_put(LED_PIN, 1);
    
    //SET GPIO pin for receiving data signals from CC1101 ASYNCHRONIOUS receiver
    gpio_set_dir(RF_DATA_IN, GPIO_IN);
    gpio_init(RF_DATA_IN);
    
    //Just a GPIO can be used as output for Oscilloscope triggering
    gpio_init(TEST_OUT);
    gpio_set_dir(TEST_OUT, GPIO_OUT);
    gpio_put(TEST_OUT, 0);
    
    BIT_POINTER = 31;
    WORD_POINTER=0;

    //PUT output of used PWM to check if PWM is running at 50uS
    gpio_set_function(PWM_OUT, GPIO_FUNC_PWM);

    //determine which PWM module to be used
    slice = pwm_gpio_to_slice_num(PWM_OUT);
    channel = pwm_gpio_to_channel(PWM_OUT);
    
    //clear interrupt flag for PWM counter reaching triggerlevel
    pwm_clear_irq(slice);
    pwm_set_irq_enabled(slice, true);

    //define which interrupt routine should be used on triggerlevel reached
    irq_set_exclusive_handler(PWM_IRQ_WRAP, on_pwm_wrap);

    //enable interrupt PWM counter reaching trigger level
    irq_set_enabled(PWM_IRQ_WRAP, true);

    //SET PWM to 50 us
    pwm_set_wrap(slice, 6249);

    //SET high time to 50%
    pwm_set_chan_level (slice, channel, 3125);
    pwm_set_phase_correct(slice,0);

    //disable PWM untile LEARN command uses 50us PWM trigger
    pwm_set_enabled(slice,0);

    // Buffer to store raw reads
    uint8_t data[6];

    // define which SPI block to use
    spi_inst_t *spi = spi0;
    uint8_t  REG_DEVID = 0x00;

    // Initialize CS pin high
    gpio_init(CS_PIN);
    gpio_set_dir(CS_PIN, GPIO_OUT);
    gpio_put(CS_PIN, 1);

    // Initialize SPI port at 1 MHz
    spi_init(spi, 1000 * 1000);

    // Set SPI format
    spi_set_format( spi0,   // SPI instance
                    8,      // Number of bits per transfer
                    0,      // Polarity (CPOL)
                    0,      // Phase (CPHA)
                    SPI_MSB_FIRST);

    // Initialize SPI pins
    gpio_set_function(SCK_PIN, GPIO_FUNC_SPI);
    gpio_set_function(MOSI_PIN, GPIO_FUNC_SPI);
    gpio_set_function(MISO_PIN, GPIO_FUNC_SPI);
    reg_read(spi, CS_PIN, 0x30, SPI_RX_DATA, 1);    
    
    // Send register settings to CC1101 through SPI interface write 1 byte at a time
    for(int i=0; i<24;i++)
    {
        reg_write(spi, CS_PIN,CC1101_REG_SETTINGS[i*2], CC1101_REG_SETTINGS[1+(i*2)] );
    }

    //Do a loop until the CC1101 is ready for next data
    do
    {
        reg_read(spi, CS_PIN, 0x34, SPI_RX_DATA, 1);
        SPI_RX_DATA[0]=SPI_RX_DATA[0]&0xF0;
    }  while(SPI_RX_DATA[0]!=0x10);

    //read back registers CC1101 just for checking
    reg_read(spi, CS_PIN, 0x00, SPI_RX_DATA, 31);
    
    //set power Amplifier output to 10dBm in OOK/ASK mode. only patable 0 for sending Zero and patable 1 for sending ONE
    SPI_RX_DATA[0]=0x00;
    SPI_RX_DATA[1]=0xC0;

    //SET OOK/ASK patable registers pattable 0 = 0x00/ pattable 1 = 0xC0 
    reg_write_burst_patatable(spi, CS_PIN, 0x3E,SPI_RX_DATA,2);
    
    // Read pattable registers for check of correct settings
    // read all 8 patable0 to patable 7 registers in burtsmode without changing CS signal
    // and all registers can be read
    reg_read(spi, CS_PIN, 0x3E, SPI_RX_DATA, 8);

    //the PI PICO ASYNCHRONIOUS data output pin should be set to output after initialising CC1101
    gpio_init(RF_pin);
    gpio_set_dir(RF_pin, GPIO_OUT);
    gpio_set_drive_strength(RF_pin, GPIO_DRIVE_STRENGTH_4MA);
    gpio_set_slew_rate(RF_pin, GPIO_SLEW_RATE_FAST);
    gpio_put(RF_pin, 0);

    while (true) 
    {
            //set CC1101 in receive mode
            reg_write(spi, CS_PIN,0x34, 0x00 );
            wait_for_settle_receive_mode();
            //Read saved HUMALIGHT remote control 32 bit code from flash. first 16 bit is remote serial number, second 16 bit is data/command
            read_stored_remote_info();
            for(teller=0; teller<25; teller++) 
            {
                recieve_char[teller] = 0x00;
            } 
            
            printf("Firmware version 1.0\n\n");
            printf("COMMAND?: \n");
            gpio_put(LED_PIN, 1);
            //scanf("%s", &recieve_char);
            fgets(recieve_char,24,stdin);
            printf("Received command: %s \n",recieve_char);
            gpio_put(LED_PIN, 0);
            sleep_ms(500);
            
            //Disable PWM until LEARN COMMAND is given. PWM and PWM interrupt is used to sample RF_INPUT pin with 50us cycle time
            pwm_set_enabled(slice,0);
            //string_equal = strncmp(recieve_char ,on_command ,3);
            
            if(strstr(recieve_char, on_command))
            {
                printf("Veranda ON \n");
                //set CC1101 in transmit mode
                reg_write(spi, CS_PIN,0x35, 0x00 );
                wait_for_settle_transmit_mode();
                REMOTE_DATA_16=0x0000;
                //Read saved HUMALIGHT remote control 32 bit code from flash
                read_stored_remote_info();
                //REMOTE_DATA_16=0x5F97;
                TRANSMIT_32_BIT=serial_and_dim_16_bit_to_32_bit(REMOTE_DATA_16,REMOTE_SERIAL_16);
                transmit(TRANSMIT_32_BIT);
                //set CC1101 in receive mode
                reg_write(spi, CS_PIN,0x34, 0x00 );
                wait_for_settle_receive_mode();
            }
            
            if(strstr(recieve_char,off_command))
            {
                printf("Veranda OFF\n");
                //set CC1101 in transmit mode
                reg_write(spi, CS_PIN,0x35, 0x00 );
                //Read saved HUMALIGHT remote control 32 bit code from flash
                read_stored_remote_info();
                //REMOTE_DATA_16=0x5F97;
                TRANSMIT_32_BIT=serial_and_dim_16_bit_to_32_bit(REMOTE_DATA_16,REMOTE_SERIAL_16);
                transmit(TRANSMIT_32_BIT);
                //set CC1101 in receive mode
                reg_write(spi, CS_PIN,0x34, 0x00 );
                wait_for_settle_receive_mode();
            }
            
            if(strstr(recieve_char, dim_100))
            {
                printf("Veranda Light 100%%\n");
                //set CC1101 in transmit mode
                reg_write(spi, CS_PIN,0x35, 0x00 );
                wait_for_settle_transmit_mode();
                //Read saved HUMALIGHT remote control 32 bit code from flash
                read_stored_remote_info();
                REMOTE_DATA_16=0x59EC;
                TRANSMIT_32_BIT=serial_and_dim_16_bit_to_32_bit(REMOTE_DATA_16,REMOTE_SERIAL_16);
                transmit(TRANSMIT_32_BIT);
                //set CC1101 in receive mode
                reg_write(spi, CS_PIN,0x34, 0x00 );
                wait_for_settle_receive_mode();
            }
            
            if(strstr(recieve_char,dim_75))
            {
                printf("Veranda Light 75%%\n");
                //set CC1101 in transmit mode
                reg_write(spi, CS_PIN,0x35, 0x00 );
                wait_for_settle_transmit_mode();
                //Read saved HUMALIGHT remote control 32 bit code from flash
                read_stored_remote_info();
                REMOTE_DATA_16=0x50E3;
                TRANSMIT_32_BIT=serial_and_dim_16_bit_to_32_bit(REMOTE_DATA_16,REMOTE_SERIAL_16);
                transmit(TRANSMIT_32_BIT);
                //set CC1101 in receive mode
                reg_write(spi, CS_PIN,0x34, 0x00 );
                wait_for_settle_receive_mode();
            }
            
            if(strstr(recieve_char, dim_50))
            {
                printf("Veranda Light 50%%\n");
                //set CC1101 in transmit mode
                reg_write(spi, CS_PIN,0x35, 0x00 );
                wait_for_settle_transmit_mode();
                //Read saved HUMALIGHT remote control 32 bit code from flash
                read_stored_remote_info();
                REMOTE_DATA_16=0x56E9;
                TRANSMIT_32_BIT=serial_and_dim_16_bit_to_32_bit(REMOTE_DATA_16,REMOTE_SERIAL_16);
                transmit(TRANSMIT_32_BIT);
                //set CC1101 in receive mode
                reg_write(spi, CS_PIN,0x34, 0x00 );
                wait_for_settle_receive_mode();
            }
            
            if(strstr(recieve_char, dim_25))
            {
                printf("Veranda Light 25%%\n");
                //set CC1101 in transmit mode
                reg_write(spi, CS_PIN,0x35, 0x00 );
                wait_for_settle_transmit_mode();
                //Read saved HUMALIGHT remote control 32 bit code from flash
                read_stored_remote_info();
                REMOTE_DATA_16=0x55E8;
                TRANSMIT_32_BIT=serial_and_dim_16_bit_to_32_bit(REMOTE_DATA_16,REMOTE_SERIAL_16);
                transmit(TRANSMIT_32_BIT);
                //set CC1101 in receive mode
                reg_write(spi, CS_PIN,0x34, 0x00 );
                wait_for_settle_receive_mode();
            }
            
            if(strstr(recieve_char, dim_xx ))
            {
                printf("Veranda DIM level = %c%c%c%%\n", recieve_char[3], recieve_char[4], recieve_char[5]);
                for(int i=0; i<3; i++)
                {
                    DIM_PERCENT[i] = recieve_char[i+3];
                }
                //Read saved HUMALIGHT remote control 32 bit code from flash
                read_stored_remote_info();
                DIM_LEVEL=string_to_integer();
                REMOTE_DIM_16=value_to_16_bit_dim_level(DIM_LEVEL);
                printf("dim level: %d%%: 16 bit dim value %X \n",DIM_LEVEL,REMOTE_DIM_16);
                TRANSMIT_32_BIT=serial_and_dim_16_bit_to_32_bit(REMOTE_DIM_16,REMOTE_SERIAL_16);
                //set CC1101 in transmit mode
                reg_write(spi, CS_PIN,0x35, 0x00 );
                wait_for_settle_transmit_mode();
                transmit_dim(TRANSMIT_32_BIT);
                //set CC1101 in receive mode
                reg_write(spi, CS_PIN,0x34, 0x00 );
                wait_for_settle_receive_mode();
                for(int i = 0 ; i<3; i++)
                {
                    DIM_PERCENT[i] = 0x00;
                }
            }
            
            if(strstr(recieve_char, hex_xx))
            {
                printf("Veranda DIM HEX level = %c%c%c%c\n", recieve_char[3], recieve_char[4], recieve_char[5], recieve_char[6]);
                for(int i=0; i<4; i++)
                {
                    DIM_HEX[i] = recieve_char[i+3];
                }
                //Read saved HUMALIGHT remote control 32 bit code from flash
                read_stored_remote_info();
                DIM_HEX_16_BIT = hex_string_to_16_bit(DIM_HEX);
                TRANSMIT_32_BIT=serial_and_dim_16_bit_to_32_bit(DIM_HEX_16_BIT,REMOTE_SERIAL_16);
                //set CC1101 in transmit mode
                reg_write(spi, CS_PIN,0x35, 0x00 );
                wait_for_settle_transmit_mode();
                transmit_dim(TRANSMIT_32_BIT);
                //set CC1101 in receive mode
                reg_write(spi, CS_PIN,0x34, 0x00 );
                wait_for_settle_receive_mode();
                for(int i =0; i<5; i++)
                {
                    DIM_HEX[i] = 0x00;
                }
            }
            if(strstr(recieve_char, down))
            {
                printf("Veranda down DIM\n");
                //Read saved HUMALIGHT remote control 32 bit code from flash
                read_stored_remote_info();
                DIM_LEVEL=100;
                //set CC1101 in transmit mode
                reg_write(spi, CS_PIN,0x35, 0x00 );
                wait_for_settle_transmit_mode();
                for(DIM_LEVEL=100;(DIM_LEVEL>=0)&&(DIM_LEVEL<=100); DIM_LEVEL=DIM_LEVEL-10)
                {
                    REMOTE_DIM_16=value_to_16_bit_dim_level(DIM_LEVEL);
                    printf("dim level: %d%%: 16 bit dim value %X \n",DIM_LEVEL,REMOTE_DIM_16);                
                    TRANSMIT_32_BIT=serial_and_dim_16_bit_to_32_bit(REMOTE_DIM_16,REMOTE_SERIAL_16);
                    transmit_fast_dim(TRANSMIT_32_BIT); 
                }
                //set CC1101 in receive mode
                reg_write(spi, CS_PIN,0x34, 0x00 );
                wait_for_settle_receive_mode();
            }
            
            if(strstr(recieve_char, up))
            {
                printf("Veranda up DIM\n");
                //Read saved HUMALIGHT remote control 32 bit code from flash
                read_stored_remote_info();
                DIM_LEVEL=0;
                //set CC1101 in transmit mode
                reg_write(spi, CS_PIN,0x35, 0x00 );
                wait_for_settle_transmit_mode();
                for(DIM_LEVEL=0; DIM_LEVEL<=100; DIM_LEVEL=DIM_LEVEL+10)
                {
                    REMOTE_DIM_16=value_to_16_bit_dim_level(DIM_LEVEL);
                    printf("dim level: %d%% : 16 bit dim value %X \n",DIM_LEVEL,REMOTE_DIM_16);                   
                    TRANSMIT_32_BIT=serial_and_dim_16_bit_to_32_bit(REMOTE_DIM_16,REMOTE_SERIAL_16);
                    transmit_fast_dim(TRANSMIT_32_BIT); 
                }
                //set CC1101 in receive mode
                reg_write(spi, CS_PIN,0x34, 0x00 );
                wait_for_settle_receive_mode();
            }
            
            if(strstr(recieve_char, learn))
            {
                printf("Learn remote control HUMALIGHT. \n\nPRESS ON/OFF BUTTON Remote Control\n");
                 //set CC1101 in receive mode
                reg_write(spi, CS_PIN,0x34, 0x00 );
                wait_for_settle_receive_mode();
                
                printf("LEARNING REMOTE\n\nAFTER 10 SECONDS IT WILL BE STOPPED\n");
                strcpy(REMOTE_SERIAL_HEX, "0000\0");
                pwm_set_enabled(slice,1);
                for(uint32_t i=0; i < 0xFFFFFFF; i++);
                printf("LEARNING REMOTE STOPPED\n");
                pwm_set_enabled(slice,0);
                
                if(strspn(REMOTE_SERIAL_HEX,"0000") == 4)
                {
                    printf("\nREMOTE SERIAL is 0000\n\nREDO LEARN ACTION !!\n" );
                }
                else 
                {
                    if((strspn(REMOTE_SERIAL_FLASH,REMOTE_SERIAL_HEX)) == 4)
                    {
                        printf("\n SERIAL %s already stored in non violent memory",REMOTE_SERIAL_HEX);
                    }
                    else
                    {
                        printf("New serial %s from remote will be stored in non violent memory\n", REMOTE_SERIAL_HEX );
                        strcpy(REMOTE_SERIAL_FLASH, REMOTE_SERIAL_HEX);
                        for(int i=0; i<256; i++)
                        {
                            FLASH_BUFFER[i]= 0xFF;
                        }
                        
                        //
                        //store 16 bit serial and 16 bit data from remote ON BUTTON into buffer for programming into flash
                        //
                        
                        FLASH_BUFFER[0]=REMOTE_SERIAL_HEX[0];
                        FLASH_BUFFER[1]=REMOTE_SERIAL_HEX[1];
                        FLASH_BUFFER[2]=REMOTE_SERIAL_HEX[2];
                        FLASH_BUFFER[3]=REMOTE_SERIAL_HEX[3];
                        FLASH_BUFFER[4]=REMOTE_DATA_HEX[0];
                        FLASH_BUFFER[5]=REMOTE_DATA_HEX[1];
                        FLASH_BUFFER[6]=REMOTE_DATA_HEX[2];
                        FLASH_BUFFER[7]=REMOTE_DATA_HEX[3];
                        FLASH_BUFFER[8]=0x00;
                        ADDRESS_FLASH_STORE=XIP_BASE + FLASH_TARGET_OFFSET;
                        POINTER_TO_FLASH=(uint32_t *)ADDRESS_FLASH_STORE;
                        printf("\nStart Serial Number programming into flash\n\n");
                        uint32_t ints = save_and_disable_interrupts();
                        flash_range_erase((PICO_FLASH_SIZE_BYTES-FLASH_SECTOR_SIZE), FLASH_SECTOR_SIZE);
                        flash_range_program((PICO_FLASH_SIZE_BYTES - FLASH_SECTOR_SIZE), (uint8_t *)FLASH_BUFFER, FLASH_PAGE_SIZE);
                        restore_interrupts (ints);
                        printf("Programmed serial Number into flash\n\n ");

                        //Read saved HUMALIGHT remote control 32 bit code from flash
                        read_stored_remote_info();
                       
                    }
                }
                
            }
            
            if(strstr(recieve_char,setting))
            {
                printf("SETTINGS\n\n");
                printf("flash size: %d \nflash sector size: %d\nflash page size: %d\n\n", PICO_FLASH_SIZE_BYTES, FLASH_SECTOR_SIZE, FLASH_PAGE_SIZE);
                //Read saved HUMALIGHT remote control 32 bit code from flash
                read_stored_remote_info();
                printf("HAMULIGHT Remote Controller info stored in FLASH.\n\n ON/OFF BUTTON Serialnumber 16 bit %X : 16 bit data: %X\n",REMOTE_SERIAL_16,REMOTE_DATA_16);
            }
            pwm_set_enabled(slice,0);
            printf("\n");  
    }

}
